/*
 * Arduino Einfach - die Arduinoseite
 * Programm wartet auf Kontakt ueber USB oder Bluetooth,
 * nimmt Befehle als Programm oder zur sofoertigen 
 * Ausfuehrung entgegen.
 *
 * LBZ & FKG Goettingen
 * Andreas Flemming, andreas.flemming@fkggoettingen.de
 * Lizenz: GPL (www.gnu.org)
 *
 * Dies ist Version 3, geeignet fuer die klassische 
 * Programmierumgebung und die Arbeit mit Zustaenden.
 * A - klassisch
 * B - Zustaende
 *
 * Weiterhin haben sich folgende Dinge gendert:
 * - Datenuebertragung als String (v.a. wegen BTSerial fuer Phonegap)
 * - Kein Speichern mehr (Warte-Problematik)
 * - Alle D-Pins (2 bis 10) als Ausgabe, Eingabe erfolgt sowieso immer
 *   ueber Analog-Pins
*/

#include <SoftwareSerial.h>
#include <Servo.h>

#define TEST_LED      5

SoftwareSerial blueSerial(11, 12);
Servo servo;


#define VM_NICHTS       1
#define VM_USB          2
#define VM_BLUETOOTH    3
int verbindungsModus = VM_NICHTS;

/*
 * Es gibt verschiedene Modi:
 *    - Sofort Ausführen - klassisch
 *    - sofort ausfuehren - mit Zustaenden
 *    - interaktiv
 */
#define AM_NICHTS                0
#define AM_PROGRAMMABARBEITEN    1
#define AM_INTERAKTIV            2
#define AM_ENDE                  3
int arbeitsModus = AM_NICHTS;

/*
 * Die Befehle:
 * s - Schicke Sensor-Daten
 * k - klopfen
 * a - alles aus
 * i - interaktiv
 * r - run, also Programm abarbeiten
 * d - debug?
 * x - Befehl ausfuehren Typ A (eXecute)
 * m - Befehl TypA speichern (memory)
 */
 
#define BF_ANKLOPFEN                     'k'  // Klopfen
#define BF_INTERAKTIV                    'i'  // Interaktiv
#define BF_PROGRAMMABARBEITEN            'r'  // Run
#define BF_GIB_SENSORDATEN               's'  // Sensordaten
#define BF_SPEICHERE_BEFEHL_A            'm'  // Memory
#define BF_SPEICHERE_BEFEHL_B            'n'  // Memory B
#define BF_LOESCHE_PROGRAMM              'l'  // Loeschen
#define BF_BEFEHLABARBEITEN_A            'x'  // eXecute
#define BF_BEFEHLABARBEITEN_B            'y'  // eXecute B
#define BF_SENDEDEBUG                    'd'  // Debug
#define BF_ALLES_AUS                     'a'  // Alles aus


/*
 * Die klassischen Befehle
 */
 
// Die Bedingungen
#define COND_IMMER               32
#define COND_NIE                 33
#define COND_WENN                34
#define COND_KLEINER             0
#define COND_GROESSER            1

// Die Befehle
#define CMD_ANFANG               0
#define CMD_SETZE_OUTPUT         1
#define CMD_SETZE_INPUT          2
#define CMD_SETZE_HIGH           3
#define CMD_SETZE_LOW            4
#define CMD_SETZE_WERT           5
#define CMD_WARTE                6
#define CMD_TON                  7
#define CMD_TON_AUS              8
#define CMD_MOTOR                9

#define MAX_INPUT 15;
byte eingabeBuffer[15];
int eingabePos = 0;

// Extrem simples Befehlsarray als Programmspeicher.
// Bedingung Bed1 Verknuepfung Bed2 Befehl
#define OFFSET_BEDINGUNG     0    // COND_IMMER, COND_NIE, COND_WENN
#define OFFSET_WER           1    // AnalogIn
#define OFFSET_RELATION      2    // 0 kleiner, 1 groesser, noch kein gleich
#define OFFSET_WERT          3    // Vergleichswert
#define OFFSET_VERKNUEPFUNG  4    // 0 - nix, 1 - oder, 2 - und
#define OFFSET_WER2          5
#define OFFSET_RELATION2     6
#define OFFSET_WERT2         7
#define OFFSET_BEFEHL        8    // Siehe CMD...
#define OFFSET_PARA1         9    // z.B. AusgabeBit
#define OFFSET_PARA2        10    // z.B. Wert, immer je nach Befehl
#define CMD_SIZE            11

#define ANZAHL_PROG_SLOTS_A 40
#define ANZAHL_PROG_SLOTS_B 40  
byte programmSpeicherA[ANZAHL_PROG_SLOTS_A][CMD_SIZE];
byte programmSpeicherB[ANZAHL_PROG_SLOTS_B][CMD_SIZE + 2]; // Zustand + Folgezustand
int freierPlatzA = 0;
int freierPlatzB = 0;
int befehlsZaehler = 0;
int programmArt = 0; // 0 - kein Programm, 1 - klassisch, 2 - mit Zustaenden
int aktuellerBefehl = 0;
int aktuellerZustand = 0;
boolean langerBefehl = false;
byte befehlsBuffer[CMD_SIZE];

// Wenn auf einem Pin die Tonausgabe aktiv ist, dann klappt ein normales analogWrite
// nicht mehr. Der Ton muss erst ausgeschaltet werden. Also Ton-Pins merken
boolean tonAusgabe[14];
int motorNummer = 0;

//int maxDebug = 400;
//byte debugDaten[400];
//int debugZaehler = 0;

void setup() {
    
    // Nur beim Start wird geprüft, wie die Verbindung
    // hergestellt wird.
    Serial.begin(115200);
    blueSerial.begin(9600); // Notfalls durchtesten
    
    verbindungsModus = VM_NICHTS;
    eingabePos = 0; 
    
    // Alle Pins auf Ausgabe,  Alle Toene aus
    for (int i = 2; i < 11; i++)   
    {
        pinMode(i, OUTPUT);
        tonAusgabe[i] = false;
    }
    
    //for (int j = 0; j < maxDebug; j++) debugDaten[j] = 255;
    digitalWrite(TEST_LED, LOW);
}

void loop() {
  
  // Wenn noch keine Verbindung existiert, dann kann man sich
  // jederzeit verbinden. Momentan kann eine Verbindung nur durch
  // Neustart des Arduino aufgeloest werden.
  if (VM_NICHTS == verbindungsModus)
  {
      if (Serial.available())
      {
          String data = Serial.readStringUntil('\n'); 
          if (data.charAt(0) == 'k')
          {
              verbindungsModus = VM_USB;
              
              // OK senden
              Serial.println("usb"); 
              
              // Wenn sich jemand verbinden will, sollte das
              // automatische Programmabarbeiten ausgeschaltet
              // werden.
              arbeitsModus = AM_INTERAKTIV;
              loescheProgramm();
              schalteAllesAus();
          }
      }
      if (blueSerial.available())
      {
          String data = blueSerial.readString();
          if (data.charAt(0) == 'k')
          {
              verbindungsModus = VM_BLUETOOTH; 
              
              // OK senden
              blueSerial.println("bt");
              
              // Wenn sich jemand verbinden will, sollte das
              // automatische Programmabarbeiten ausgeschaltet
              // werden.
              arbeitsModus = AM_INTERAKTIV;
              loescheProgramm();
              schalteAllesAus();
          }
      }
  }
 
  String daten;
  boolean datenGelesen = false;
  if (VM_USB == verbindungsModus)
  {
      // USB?
      if (Serial.available())
      {
          daten = Serial.readStringUntil('\n');
          datenGelesen = true;
      }
  }
  else if (VM_BLUETOOTH == verbindungsModus)
  {
      if (blueSerial.available())
      {
          daten = blueSerial.readStringUntil('\n');
          datenGelesen = true;
      }
  }
  
  if (false == datenGelesen)
  {
      // Es wurde kein Byte gelesen. Ist man im Programmabarbeitungsmodus
      // ist dies voellig ok.
      if (AM_PROGRAMMABARBEITEN == arbeitsModus) 
      {
          arbeiteProgrammAb();
      }
  }
  else 
  {    
      //if (debugZaehler < maxDebug) debugDaten[debugZaehler++] = datenByte;
      
      // Daten kommen rein als String. Das ersteZeichen ist der Befehl
      char befehl = daten.charAt(0);
      switch (befehl)
      {
          case 'k': // Anklopfen, keine Parameter
              if (verbindungsModus == VM_USB)
                {
                    Serial.println("usb"); 
                    
                    // Wenn sich jemand verbinden will, sollte das
                    // automatische Programmabarbeiten ausgeschaltet
                    // werden.
                    arbeitsModus = AM_INTERAKTIV;
                    loescheProgramm();
                    schalteAllesAus();
                }
                else if (verbindungsModus == VM_BLUETOOTH)
                {
                    blueSerial.println("bt");

                    // Wenn sich jemand verbinden will, sollte das
                    // automatische Programmabarbeiten ausgeschaltet
                    // werden.
                    arbeitsModus = AM_INTERAKTIV;
                    loescheProgramm();
                    schalteAllesAus();
                }
              break;
              
          case 's': // Sensordaten sind gewuenscht
              
              schickeSensorWerte(); 
              break;
          
          case 'i': // interaktiv
          
              arbeitsModus = AM_INTERAKTIV; 
              break;
      
          case 'r': // run, Programm abarbeiten
          
              arbeitsModus = AM_PROGRAMMABARBEITEN;
              befehlsZaehler = 0;
              // Startzustand setzen?
              aktuellerZustand = 0; // muss spaeter als Para uebertragen werden
              break;
              
          case 'a': // Alles aus
              
              schalteAllesAus();
              break;

          case 'x': // Befehl sofort ausfuehren
              
              zerlegeBefehl(daten.substring(1, daten.length()), befehlsBuffer);
              fuehreBefehlAus(befehlsBuffer, true);
              break;
              
          case 'l':
          
              arbeitsModus = AM_INTERAKTIV;
              schalteAllesAus();
              loescheProgramm();
              break;
              
          case 'm': // Befehl A speichern
              
              // Wenn die bisherige Programmart anders war, wird umgeschaltet!
              if (programmArt == 2)
              {
                  freierPlatzA = 0;
              }                      
              programmArt = 1;
              zerlegeBefehl(daten.substring(1, daten.length()), befehlsBuffer);
              fuegeBefehlHinzuA(befehlsBuffer);
              break;
      }
  }
}

/*
 * Zerlegt einen String in einen Typ-A-Befehl
 * Es gibt keine Trennzeichen, die Laenge ist immer - notfalls mit
 * fuehrenden Nullen - fest!
 * Aufbau:
 * (i|w|v)[Eingang Relation Wert [Verknuepfung Eingang Relation Wert]] Befehl Para1 Para2
 * Eingang 0..3
 * Para, Wert 000..255
 * Verknpuefung u oder o
 * Relation < oder >
 * Beispiele
 * i5005255            immer Lampe(5) 2(05) auf 255
 * i9006040            immer motor(9) 1 auf 40
 * w1<1007003070       wenn Eingang2 < 100 Dann Ton(7) 1 auf 70
 * v1<030u2>0307003070 wenn Eingang2 < 30 und Eingang 3 > 30 dann Ton(7) 1 auf 70
 * zwei Bedingungen mit v kodieren. Analyse geht einfacher
 */
void zerlegeBefehl(String s, byte b[])
{
    // Leider gibt es kein split, also durchgehen bis length()
    int pos = 0;
    String hilfsString;
    
    // immer oder wenn
    char z = s.charAt(pos); 
    pos++;  
    if (z == 'w')
    {
        // eine Bedingung
        b[0] = COND_WENN;
        
        // ein Zeichen Wer (0..3), dann die Bedingung und drei Zeichen was
        b[1] = (byte) (s.charAt(pos) - 48);
        pos++;
        
        if (s.charAt(pos) == '<')
            b[2] = 0;
        else
            b[2] = 1;
        pos++;
            
        hilfsString = s.substring(pos, pos + 3);
        b[3] = (byte) hilfsString.toInt();
        pos += 3;
        
        b[4] = 0;
    }
    else if (z == 'v')
    {
        // zwei Bedingungen
        b[0] = COND_WENN;
        
        // ein Zeichen Wer (0..3), dann die Bedingung und drei Zeichen was
        b[1] = (byte) (s.charAt(pos) - 48);
        pos++;
        
        if (s.charAt(pos) == '<')
            b[2] = 0;
        else
            b[2] = 1;
        pos++;
            
        hilfsString = s.substring(pos, pos + 3);
        b[3] = (byte) hilfsString.toInt();
        pos += 3;
        
        b[4] = (s.charAt(pos) == 'u') ? 1 : 2 ;
        pos++;
        
        // ein Zeichen Wer (0..3), dann die Bedingung und drei Zeichen was
        b[5] = (byte) (s.charAt(pos) - 48);
        pos++;
        
        if (s.charAt(pos) == '<')
            b[6] = 0;
        else
            b[6] = 1;
        pos++;
            
        hilfsString = s.substring(pos, pos + 3);
        b[7] = (byte) hilfsString.toInt();
        pos += 3;
    }
    else if (z == 'i')
    {
        // keine Bedingung
        b[0] = COND_IMMER;
    }
    //if (z=='i') digitalWrite(TEST_LED, HIGH);
    
    // Der Befehl ist eine Ziffer von 0 bis 9, also ein Zeichen
    z = s.charAt(pos);
    b[8] = (int) z - 48;
    pos++;
    
    // Para1 000 ... 255
    b[9] = s.substring(pos, pos + 3).toInt();
    pos += 3;
    
    // Para2 000 ... 255
    b[10] = s.substring(pos, pos + 3).toInt();
    
    // Evtl. weitere Befehle im Expertenmodus
}

/*
 * Arbeite das aktuelle Programm ab.
 * Testbetrieb: Programmspeicher ist einfaches Array und fluechtig.
 * Damit der Modus umschaltbar bleibt, wird immer nur ein Befehl
 * abgearbeitet
 */
void arbeiteProgrammAb()
{
    if (1 == programmArt)
    {
        // Klassiches Programm. Ein Befehlszaehler gibt den
        // aktuellen Stand an. Pro Durchlauf ein Befehl
        if (befehlsZaehler < freierPlatzA)
        {
            fuehreBefehlAus(programmSpeicherA[ befehlsZaehler ], false);
            befehlsZaehler++;
        }
        else
        {
            // Dann wieder von vorn
            befehlsZaehler = 0;
        }
    } 
    else if (2 == programmArt)
    {
        // Zustandsbasiert. Es gibt einen aktuellen Zustand. Zuerst
        // kommt der Zustand. Dann kommen elf Bytes des kjlassischen Befehls.
        // dann kommt der Folgezustand, der eingenommen wird, wenn die
        // Bedingung wahr war.
        int i = 0; 
        
        while (i < freierPlatzB)
        {
            int state = programmSpeicherB[i][0];
            if (state == aktuellerZustand)
            {
                // Der Befehl gehoert zum aktuellen Zustand
                // und darf ausgefuehrt werden.
                byte pA[CMD_SIZE];
                for (int j = 0; j < CMD_SIZE; j++)
                {
                    pA[j] = programmSpeicherB[i][j + 1]; 
                    //Serial.print(pA[j]); Serial.print(" ");
                }
                //Serial.println();
                    
                boolean result = fuehreBefehlAus(pA, false);
                if (result)
                {
                    // Dann gab es einen Zustandswechsel
                    aktuellerZustand = programmSpeicherB[i][CMD_SIZE+1];
                    break;
                }
            }
            i++;
        }
    }
}


/*
 * Der Befehl wird dem Programm hinzugefügt.
 */
void fuegeBefehlHinzuA(byte eingabe[])
{
    int i = 0;
    while (i < CMD_SIZE)
    {
        programmSpeicherA[freierPlatzA][i] = eingabe[i];
        i++;
    }
    freierPlatzA++;
}

/*
 * Der Befehl wird dem Programm hinzugefügt.
 */
void fuegeBefehlHinzuB(byte eingabe[])
{
    int i = 0;
    while (i < CMD_SIZE + 2)
    {
        programmSpeicherB[freierPlatzB][i] = eingabe[i];
        i++;
    }
    freierPlatzB++;
}

/*
 * Alle acht Bytes sammeln. Dann ausfuehren
 */
void sofortAusfuehren(byte datenByte)
{
    eingabeBuffer[ eingabePos ] = datenByte;
    eingabePos++;
    if (CMD_SIZE == eingabePos)
    {
        // Befehl wurde vollstaendig übertragen, nun auswerten
        fuehreBefehlAus(eingabeBuffer, true);
        eingabePos = 0;
        aktuellerBefehl = 0;
        langerBefehl = false;
    }
}

/*
 * Das konkrete Ausfuehren des Befehls. Nutzt sowohl
 * die Fernbedienung als auch die Programmabarbeitung.
 *
 * Rueckgabe: true, wenn ein Befehl auch wirklich ausgefuehrt
 * wurde.
 */
boolean fuehreBefehlAus(byte buffer[], boolean fernBedienung)
{
    // Slot leer?
    if (buffer[0] == 0) return false;
    
    // Bei Fernbedienung muss die Bedingung nicht getestet werden
    boolean machWas = false;
    if (fernBedienung)
    {
        machWas = true;
    }
    else
    {
        if (COND_IMMER == buffer[OFFSET_BEDINGUNG]) machWas = true;
        if (COND_NIE   == buffer[OFFSET_BEDINGUNG]) machWas = false;
        if (COND_WENN  == buffer[OFFSET_BEDINGUNG]) 
        {
            boolean machWas1 = false, machWas2 = false;
            int gelesen =  map( analogRead(buffer[OFFSET_WER]), 0, 1024, 0, 255);
            int wert    = buffer[OFFSET_WERT];

            if (buffer[OFFSET_RELATION] == COND_KLEINER)
            {
                // Kleiner
                if (gelesen < wert) machWas1 = true; else machWas1 = false;
            }
            else
            {
                // groesser
                if (gelesen > wert) machWas1 = true; else machWas1 = false;
            }
                        
            // Eine zweite Bedingung?
            if (buffer[OFFSET_VERKNUEPFUNG] != 0)
            {
                gelesen =  map( analogRead(buffer[OFFSET_WER2]), 0, 1024, 0, 255);
                wert    = buffer[OFFSET_WERT2];
                if (buffer[OFFSET_RELATION2] == COND_KLEINER)
                {
                    // Kleiner
                    if (gelesen < wert) machWas2 = true; else machWas2 = false;
                }
                else
                {
                    // groesser
                    if (gelesen > wert) machWas2 = true; else machWas2 = false;
                }
            }
            
            // Nun das machWas zusammenbasteln
            switch (buffer[OFFSET_VERKNUEPFUNG])
            {
                case 0: machWas = machWas1;
                        break;
                case 1: machWas = machWas1 || machWas2;
                        break;
                case 2: machWas = machWas1 && machWas2;
                        break;
            }
        }
    }
    
    if (!machWas) return false;
    
    int zaehler = 0;
    switch (buffer[OFFSET_BEFEHL])
    {
        case CMD_SETZE_OUTPUT: 
                               pinMode( buffer[OFFSET_PARA1], OUTPUT);
                               break;
        case CMD_SETZE_INPUT:  
                               pinMode( buffer[OFFSET_PARA1], INPUT);
                               break;
        case CMD_SETZE_HIGH:   
                               if (tonAusgabe[ buffer[OFFSET_PARA1] ])
                               {
                                     noTone(buffer[OFFSET_PARA1]);
                                     tonAusgabe[ buffer[OFFSET_PARA1] ] = false;
                               }
                               if (motorNummer == buffer[OFFSET_PARA1] ) motorNummer = 0;
                               digitalWrite( buffer[OFFSET_PARA1], HIGH); 
                               break;
        case CMD_SETZE_LOW:    
                               if (tonAusgabe[ buffer[OFFSET_PARA1] ])
                               {
                                     noTone(buffer[OFFSET_PARA1]);
                                     tonAusgabe[ buffer[OFFSET_PARA1] ] = false;
                               }
                               if (motorNummer == buffer[OFFSET_PARA1] ) motorNummer = 0;
                               digitalWrite( buffer[OFFSET_PARA1], LOW);
                               break;
        case CMD_SETZE_WERT:   
                               if (tonAusgabe[ buffer[OFFSET_PARA1] ])
                               {
                                     noTone(buffer[OFFSET_PARA1]);
                                     tonAusgabe[ buffer[OFFSET_PARA1] ] = false;
                               }
                               if (motorNummer == buffer[OFFSET_PARA1] ) motorNummer = 0;
                               analogWrite( buffer[OFFSET_PARA1], buffer[OFFSET_PARA2]);
                               break;
        case CMD_MOTOR: // Die Motorwerte mssen speziell berechnet werden.   
                               if (tonAusgabe[ buffer[OFFSET_PARA1] ])
                               {
                                     noTone(buffer[OFFSET_PARA1]);
                                     tonAusgabe[ buffer[OFFSET_PARA1] ] = false;
                               }
                               servo.attach( buffer[OFFSET_PARA1]  );
                               servo.write( buffer[OFFSET_PARA2] );
                               // Fuer das Ausschalten muss auch der Motor gemerkt werden
                               motorNummer = buffer[OFFSET_PARA1];
                               break;
        case CMD_TON:          
                               tone( buffer[OFFSET_PARA1], 10 * buffer[OFFSET_PARA2]);
                               tonAusgabe[ buffer[OFFSET_PARA1] ] = true;
                               if (motorNummer == buffer[OFFSET_PARA1] ) motorNummer = 0;
                               break;
        case CMD_TON_AUS:      
                               noTone(buffer[OFFSET_PARA1]);
                               break;
        case CMD_WARTE:        
                               delay( 100 * buffer[OFFSET_PARA1]); 
                               break;
        default:               //testBlinken(); 
                               break;
    }
    
    return true;
}

void schickeSensorWerte()
{
    String werte = "";
    werte = werte + analogRead(0) + "," + analogRead(1) + "," + 
                    analogRead(2) + "," + analogRead(3);
    if (verbindungsModus == VM_BLUETOOTH)
    {
        blueSerial.println(werte);
    }
    if (verbindungsModus == VM_USB)
    {
        Serial.println(werte);
    }
}

void testBlinken()
{
    int x = 4;
    pinMode(TEST_LED, OUTPUT);
    while (x > 0)
    {
        digitalWrite(TEST_LED, HIGH);
        delay(150);
        digitalWrite(TEST_LED, LOW);
        delay(150);
        x--;
    }
}

void numBlinken(int n)
{
    pinMode(TEST_LED, OUTPUT);
    while (n > 0)
    {
        digitalWrite(TEST_LED, HIGH);
        delay(250);
        digitalWrite(TEST_LED, LOW);
        delay(250);
        n--;
    }
    delay(2000);
}

void schalteAllesAus()
{
    for (int i = 2; i < 14; i++)
    {
        if (tonAusgabe[i])
        {
            tonAusgabe[i] = false;
            noTone(i);
        }
        else
        {
            digitalWrite(i, 0);
        }
    }
    if (motorNummer > 0) 
    {
        servo.attach(motorNummer);
        servo.write(0);
    }
}

void loescheProgramm()
{
    // Auch alle alten Befehle loeschen
    if (1 == programmArt)
    {
        for (int i = 0; i < ANZAHL_PROG_SLOTS_A; i++)
        {
            for (int j = 0; j < CMD_SIZE; j++)
            {
                programmSpeicherA[i][j] = 0;
            }
        }                  
    }
    else if (2 == programmArt)
    {
        for (int i = 0; i < ANZAHL_PROG_SLOTS_B; i++)
        {
            for (int j = 0; j < CMD_SIZE + 2; j++)
            {
                programmSpeicherB[i][j] = 0;
            }
        }
    }
    
    freierPlatzA = 0;
    freierPlatzB = 0; 
}


